/* JMC - JuggleMaster Pattern Converter
 *
 * Version 1.0
 * Copyright (C) Per Johan Persson 2001
 *
 *
 * data.c - Holds all data structures and data. All methods
 *          for accessing data are here.
 */

#include "data.h"

#define DEFAULT_DR 0.50
#define DEFAULT_HR 0.20

int is_big_endian(void) {
  int x = 1;
  if(*(char *)&x == 1)
    return FALSE;
  else
    return TRUE;
}

/* fixme: adding any category except normal _without_
   adding data should produce a fatal error */

/* set whenever it is legal to add pattern data */
BOOL add_patt_data_ok = FALSE;

/* current value of dwell ratio and height ratio */
float curr_dr = (float)DEFAULT_DR, curr_hr = (float)DEFAULT_HR;

/* current style length */
int curr_stylelen = 0;

/* parameters */
void set_dr(float value) {
  if (value <= 0.0) {
    warning("Illegal value for dwell ratio, using default\n");
    curr_dr = DEFAULT_DR;
  }
  else
    curr_dr = value;
}

void set_hr(float value) {
  if (value <= 0.0) {
    warning("Illegal value for height ratio, using default\n");
    curr_hr = DEFAULT_DR;
  }
  else
    curr_hr = value;
}

/* the following parameters can only be set once */
void set_ga(float value) {}
void set_sp(float value) {}
void set_bp(int value) {}
void set_hd(int value) {}
void set_pd(int value) {}
void set_mr(int value) {}

/************************************************
 ** Data structure creation, called from parser
 ************************************************/

/* Set the style. If it isn't present, add it, otherwise
   set the pointer to it (current_style) */
/* FIXME: Currently unsafe */
void set_style(char* name) {
  int i;

  //printf("Setting style to '%s' (line %d)\n", name, lineno);

  if (strlen(name) > JML_MAX_NAMELEN) {
    warning("Name too long, truncating (%d)\n", strlen(name));
    name[JML_MAX_NAMELEN-1] = '\0';
  }

  /* Never add the normal style */
  /* fixme: should set specific error message when
     attempting to define the normal style, or,
     set_style("Normal") should be called in init */
  if (strcmp(name, "Normal") == 0) {
    current_style_pos = -1;
    return;
  }

  /* first check it the style is already present,
     in which case adding style data to it is illegal. */
  for (i=0; i<style_size; i++) {
    if (strcmp(styles[i]->name, name) == 0) {
      /* Style already present */
      current_style = styles[i];
      current_style_pos = i;
      /* Adding data to an existing style is illegal */
      add_patt_data_ok = FALSE;
      return;
    }
  }

  /* style is not present, add it */
  styles[style_size] = (styledef*)xmalloc(sizeof(styledef));
  styles[style_size]->id = ID_STYLE;
  strcpy(styles[style_size]->name, name);
  styles[style_size]->offset = styled_size/4;
  styles[style_size]->length = 0;

  add_patt_data_ok = TRUE;
  current_style_pos = style_size;
  current_style = styles[style_size++];

  curr_stylelen = 0;
}

/* FIXME: Currently unsafe */
void add_to_style(int d1, int d2, int d3, int d4) {
  //printf("Styledata: {%d, %d}{%d, %d}\n", d1, d2, d3, d4);
  
  /* Ok to add style data? */
  if (add_patt_data_ok) {
    styledata[styled_size++] = (char)d1;
    styledata[styled_size++] = (char)d2;
    styledata[styled_size++] = (char)d3;
    styledata[styled_size++] = (char)d4;
    current_style->length++;
  }
  else
    error("Can't add data to an already defined style\n");

  curr_stylelen++;
  if (curr_stylelen > JML_MAX_STYLELEN)
    error("Style length exceeded, maximum no of entries is %d\n",
	  JML_MAX_STYLELEN);
}

/* fixme: Trim category name */
void set_category(char* name) {
  int i;

  if (strlen(name) > JML_MAX_NAMELEN) {
    warning("Name too long, truncating (%d)\n", strlen(name));
    name[JML_MAX_NAMELEN-1] = '\0';
  }

  /* First set current pattern's next to -1
     unless this is the first category set */
  if (current_cat != NULL) {
    if (current_patt == NULL) {
      error("No patterns in category\n");
      return;
    }
    current_patt->next_pattern = -1;
  }

  /* Check if the category is already present, in 
     which case an error should be issued */
  for (i=0; i<cat_size; i++) {
    if (strcmp(categories[i]->name, name) == 0) {
      error("Category %s already present, ignoring\n", name);
      return;
    }
  }

  /* category is not present, add it */
  categories[cat_size] = (categorydef*)xmalloc(sizeof(categorydef));
  if (current_patt_pos == 0)
    categories[cat_size]->first_patt = 0;
  else
    categories[cat_size]->first_patt = current_patt_pos+1;

  strcpy(categories[cat_size]->name, name);

  current_cat = categories[cat_size++];
}

/* fixme: Trim pattern name */
void add_pattern(char* site, char* name) {
  int i;

  //printf("Adding pattern '%s' with site '%s'\n", name, site);

  if (strlen(name) > JML_MAX_NAMELEN) {
    warning("Name too long, truncating (%d)\n", strlen(name));
    name[JML_MAX_NAMELEN-1] = '\0';
  }

  if (strlen(site) > JML_MAX_SITELEN) {
    error("Siteswap too long (%d)\n", strlen(site));
    return;
  }

  /* Adding a pattern without setting category is
     a fatal error */
  if (current_cat == NULL) {
    error("Can't add patterns before setting category, ignoring\n");
    return;
  }

  /* fixme: the pattern "3 cascade" is built in */

  /* first check if the pattern's name is already present,
     in which case the program should quit with an error */
  for (i=0; i<patt_size; i++) {
    if (strcmp(patterns[i]->name, name) == 0) {
      warning("Pattern already present, ignoring\n");
      return;
    }
  }

  /* pattern is not present, add it */
  patterns[patt_size] = (patterndef*)xmalloc(sizeof(patterndef));
  patterns[patt_size]->id = ID_PATTERN;
  strcpy(patterns[patt_size]->name, name);
  strcpy(patterns[patt_size]->site, site);
  patterns[patt_size]->hr = (UInt32)(curr_hr*1000);
  patterns[patt_size]->dr = (UInt32)(curr_dr*1000);

  //printf("Adding pattern %s with hr %d\n", name, patterns[patt_size]->hr);
  //printf("Adding pattern %s with dr %d\n", name, patterns[patt_size]->dr);

  /*
  if (curr_dr > 0.50) {
    printf("Adding pattern %s with dr = %f and hr = %f\n",
	   name, curr_dr, curr_hr);
  }
  */

  /* special handling for normal style (-1) */
  if (current_style_pos == -1)
    patterns[patt_size]->style = 0;
  else
    patterns[patt_size]->style = current_style_pos + STYLE_OFFSET;

  patterns[patt_size]->next_pattern = patt_size+1;

  current_patt_pos = patt_size;
  current_patt = patterns[patt_size++];
}

void finalize_data(void) {
  int i;
  int pattern_offset = STYLE_OFFSET + style_size;

  current_patt->next_pattern = -1;

  for(i=0;i<cat_size;i++) {
    categories[i]->first_patt+=pattern_offset;
  }

  /* no conversion neccesary for big endian (palm is big endian) */
  if (is_big_endian()) {
    phd.pattern_count = patt_size;
    phd.style_count = style_size;
    phd.styledata_len = styled_size;
    phd.style_offset = STYLE_OFFSET;
    phd.category_count = cat_size;
    phd.pattern_offset = pattern_offset;
    phd.dbversion = 3;
    phd.record_count = STYLE_OFFSET + patt_size + style_size;
    /* set reserved data */
    memset(phd.reserved, '?', 32);
    
    for(i=0;i<style_size;i++) {
      styles[i]->length = styles[i]->length;
      styles[i]->offset = styles[i]->offset;
    }
    
    for(i=0;i<patt_size;i++) {
      patterns[i]->hr = patterns[i]->hr;
      patterns[i]->dr = patterns[i]->dr;
      patterns[i]->style = patterns[i]->style;
      /* correct offset for next_pattern */
      if (patterns[i]->next_pattern != -1)
        patterns[i]->next_pattern = patterns[i]->next_pattern+
                                          pattern_offset;
    }
    
    for(i=0;i<cat_size;i++)
      categories[i]->first_patt = categories[i]->first_patt;
  }
  /* convert for little endian platforms (e.g. intel) */
  else {
    phd.pattern_count = htons((unsigned short)patt_size);
    phd.style_count = htons((unsigned short)style_size);
    phd.styledata_len = htonl(styled_size);
    phd.style_offset = htons(STYLE_OFFSET);
    phd.category_count = htonl(cat_size);
    phd.pattern_offset = htons((unsigned short)pattern_offset);
    phd.dbversion = htonl(3);
    phd.record_count = htonl(STYLE_OFFSET + patt_size + style_size);
    /* set reserved data */
    memset(phd.reserved, '?', 32);
    
    for(i=0;i<style_size;i++) {
      styles[i]->length = htons(styles[i]->length);
      styles[i]->offset = htons(styles[i]->offset);
    }
    
    for(i=0;i<patt_size;i++) {
      patterns[i]->hr = htonl(patterns[i]->hr);
      patterns[i]->dr = htonl(patterns[i]->dr);
      patterns[i]->style = htonl(patterns[i]->style);
      /* correct offset for next_pattern */
      if (patterns[i]->next_pattern != -1)
        patterns[i]->next_pattern = htonl(patterns[i]->next_pattern+
                                          pattern_offset);
    }
    
    for(i=0;i<cat_size;i++)
      categories[i]->first_patt = htonl(categories[i]->first_patt);
  }
}


/************************************************
 ** Data structure management
 ************************************************/

/*
categorydef* categories;
patterndef*  patterns;
styledef*    styles;
patt_header  phd;
char*        styledata;

int cat_init, cat_size, patt_init, catt_size,
    style_init, style_size, styled_init, styled_size;
*/

/* Call to do first time initialization of data */
void init_data(void) {
  /* initialize styles */
  style_init = MEM_INTERVAL;
  style_size = 0;
  styles = (styledef**)xmalloc(style_init*sizeof(styledef*));

  /* initialize styledata */
  styled_init = MEM_INTERVAL;
  styled_size = 0;
  styledata = (char*)xmalloc(styled_init*sizeof(char*));

  /* initialize patterns */
  patt_init = MEM_INTERVAL;
  patt_size = 0;
  patterns = (patterndef**)xmalloc(patt_init*sizeof(patterndef*));

  /* initialize categories */
  cat_init = MEM_MIN_INTERVAL;
  cat_size = 0;
  categories = (categorydef**)xmalloc(cat_init*sizeof(categorydef*));
  
  /* reset current pointers */
  current_style = NULL;
  current_style_pos = 0;
  current_cat = NULL;
  current_patt = NULL;
  current_patt_pos = 0;
}

/* FIXME: Add functions to grow the data structures here */
/*
void grow_style(void);
void grow_styled(void);
void grow_cat(void); / use MEM_MIN_INTERVAL
void grow_patt(void);
*/

/************************************************
 ** Debugging: Print data structure
 ************************************************/

void print_styledata(void) {
  int i;

  printf("======== Style data =======\n");

  for (i=0; i< styled_size; i+=4) {
    printf("Style {%d,%d}{%d,%d}\n", styledata[i], styledata[i+1],
	   styledata[i+2], styledata[i+3]);
  }
}

void print_styledef(void) {
  int i;
  styledef* s;

  printf("======== Style definitions =======\n");

  for (i=0; i< style_size; i++) {
    s = styles[i];

    printf("Style '%s'\n", s->name);
    printf("\tid = %d\n", s->id);
    printf("\tlength = %d\n", s->length);
    printf("\toffset = %d\n", s->offset);
  }
}

void print_cat(void) {
  int i;
  categorydef* c;

  printf("======== Category definitions =======\n");

  for (i=0; i< cat_size; i++) {
    c = categories[i];

    printf("Category '%s'\n", c->name);
    printf("\tfirst_pattern = %d\n", c->first_patt);
  }
}

void print_pattdef(void) {
  int i;
  patterndef* p;

  printf("======== Pattern definitions =======\n");

  for (i=0; i< patt_size; i++) {
    p = patterns[i];

    printf("Pattern '%s'\n", p->name);
    printf("\tid = %d\n", p->id);
    printf("\tsite = '%s'\n", p->site);
    printf("\tdr = %d\n", (int)p->dr);
    printf("\thr = %d\n", (int)p->hr);
    printf("\tstyle = %d (%d)\n", p->style, p->style-STYLE_OFFSET);
    if (p->style != 0)
      printf("\t(style name = '%s')\n", styles[p->style-STYLE_OFFSET]->name);
    else
      printf("\t(style name = 'Normal')\n");


    /* add specific handling here */
    printf("\tnext pattern = %d\n", p->next_pattern);
  }
}

void print_hdr(void) {
  printf("Header\n");
  printf("\tpattern_count = %d\n", phd.pattern_count);
  printf("\tstyle_count = %d\n", phd.style_count);
  printf("\tstyledata_len = %d\n", phd.styledata_len);
  printf("\tstyle_offset = %d\n", phd.style_offset);
  printf("\tpattern_offset = %d\n", phd.pattern_offset);
  printf("\trecord_count = %d\n", phd.record_count);
  printf("\tdbversion = %d\n", phd.dbversion);

}

#ifdef _WIN32
int isblank(int c) {
  if (c == ' ' || c == '\t' || c == '\n' || c == '\r')
    return TRUE;
  else
    return FALSE;
}
#endif