/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 * Copyright 1999 David Williams. All rights reserved.
 *
 * fpopen.c
 *
 * created: djw@djw.org, January 2, 1999.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "prcp.h"

static int
read_records(prc_internal_t* iprc,
             FILE* fp,
             prc_uint32_t filesize,
             prc_uint32_t* firstRecordOffset_r)
{
    prc_uint16_t           i;
    prc_internal_record_t* foo;
    prc_internal_record_t* prev = NULL;
    int n = 0;
    unsigned isResource = PRC_IS_RESOURCE(&iprc->header);

    iprc->records = NULL;

    for (i = 0; i < iprc->header.nrecords; i++) {
        union {
            prc_record_header_t   record;
            prc_resource_header_t resource;
        } uBuf;
        int rv;

        if (isResource) {
            rv = prcfreadresource(fp, &uBuf.resource);
        } else {
            rv = prcfreadrecord(fp, &uBuf.record);
        }

        if (rv == -1)
            return rv;

#if 0
        fprintf(stderr,
                "%02d: %02d %5d\n",
                i, uBuf.record.id, uBuf.record.offset);
#endif

        foo = (prc_internal_record_t*)prcnewrecord();
        if (foo == NULL)
            return -1;

        if (isResource) {
            foo->u_contents.record.data = (prc_byte_t*)uBuf.resource.offset;
            foo->u_contents.resource.id = uBuf.resource.id;
            foo->u_contents.resource.type = uBuf.resource.type;
        } else {
            foo->u_contents.record.data = (prc_byte_t*)uBuf.record.offset;
            foo->u_contents.record.id = uBuf.record.id;
            foo->u_contents.record.flags = uBuf.record.flags;
        }
        foo->next = NULL;
        
        if (prev != NULL) {
            prev->next = foo;
        } else {
            iprc->records = foo;

            if (!isResource)
                *firstRecordOffset_r = uBuf.record.offset;
        }
        
        prev = foo;
    }
    
    /* seek to read data */
    for (foo = iprc->records; foo != NULL; foo = foo->next) {
        void*        data;
        size_t       nread;
        long         posn;
        long         next_posn;
        size_t       size;

        posn = (long)foo->u_contents.record.data;
        if (foo->next != NULL)
            next_posn = (long)foo->next->u_contents.record.data;
        else
            next_posn = filesize;

        size = next_posn - posn;

        if (size >= (64 * 1024)) {
            fprintf(stderr, ">= 64k length record (%d)\n", n);
            return -1;
        } else if (size == 0) {
            fprintf(stderr, "zero length record (%d)\n", n);
            data = NULL;
        } else {
            data = (void*)malloc(size);
            
            if (data == NULL)
                return -1;
            
            if (fseek(fp, posn, SEEK_SET) == -1) {
                fprintf(stderr, "seek failed for record %d\n", n);
                return -1;
            }
            nread = fread(data, 1, size, fp);
            
            if (size != nread) {
                fprintf(stderr, "expected %ld, read %ld\n",
                        (prc_uint32_t)size, (prc_uint32_t)nread);
            return -1;
            }
        }
            
        foo->u_contents.record.data = data;
        foo->u_contents.record.datalen = size;

        n++;
    }
    return n;
}

static void*
read_info(prc_internal_t* iprc, FILE* fp, const char* name,
          prc_uint32_t offset, prc_uint32_t size)
{
    size_t nread;
    void* data = (void*)malloc(size);

    if (!data) {
        fprintf(stderr, "memory allocation failed for %s\n", name);
        return NULL;
    }

    if (fseek(fp, (size_t)offset, SEEK_SET) == -1) {
        fprintf(stderr, "seek failed for %s\n", name);
        free(data);
        return NULL;
    }

    nread = fread(data, 1, (size_t)size, fp);
            
    if (size != (prc_uint32_t)nread) {
        fprintf(stderr, "fread failed for %s expected %ld, read %ld\n",
                name, size, (prc_uint32_t)nread);
        free(data);
        return NULL;
    }

    return data;
}

static int
read_app_info(prc_internal_t* iprc, FILE* fp,
              prc_uint32_t offset, prc_uint32_t size)
{
    void* data;

    data = read_info(iprc, fp, "appInfo", offset, size);
    if (data == NULL)
        return -1;

    iprc->header.appinfoData = data;
    iprc->header.appinfoSize = size;

    return 0;
}

static int
read_sort_info(prc_internal_t* iprc, FILE* fp,
              prc_uint32_t offset, prc_uint32_t size)
{
    void* data;

    data = read_info(iprc, fp, "sortInfo", offset, size);
    if (data == NULL)
        return -1;

    iprc->header.sortinfoData = data;
    iprc->header.sortinfoSize = size;

    return 0;
}

prc_t*
prcfpopen(FILE* fp, unsigned mode)
{
    prc_t      buf;
    prc_internal_t* iprc = NULL;
    prc_uint32_t appInfoOffset = 0;
    prc_uint32_t sortInfoOffset = 0;

    if ((mode & PRC_OPEN_CREATE) != 0) {
        prcinitheader(&buf);
    } else {
        fseek(fp, 0, SEEK_SET);
        
        if (prcfreadheader(fp, &buf, &appInfoOffset, &sortInfoOffset) != 0)
            return NULL;
    }
    
    iprc = (prc_internal_t*)malloc(sizeof(prc_internal_t));

    if (iprc == NULL)
        return NULL;

    memcpy(&iprc->header, &buf, sizeof(prc_t));
    iprc->records = NULL;
    iprc->fp = fp;
    iprc->mode = mode;

    if ((mode & PRC_OPEN_CREATE) == 0) {
        struct stat statbuf;
        int    rv;
        prc_uint32_t firstRecordOffset = 0;

        if (fstat(fileno(fp), &statbuf) == -1) {
            fprintf(stderr, "could not stat!\n");
            return NULL;
        }

        rv = read_records(iprc, fp, statbuf.st_size, &firstRecordOffset);

        if (!PRC_IS_RESOURCE(&iprc->header)) {
            if (appInfoOffset != 0) {
                prc_uint32_t appInfoSize;

                if (sortInfoOffset != 0)
                    appInfoSize = sortInfoOffset - appInfoOffset;
                else if (firstRecordOffset != 0)
                    appInfoSize = firstRecordOffset - appInfoOffset;
                else
                    appInfoSize = statbuf.st_size - appInfoOffset;

                if (read_app_info(iprc, fp, appInfoOffset, appInfoSize) == -1)
                    return NULL;
            }

            if (sortInfoOffset != 0) {
                prc_uint32_t sortInfoSize;

                if (firstRecordOffset != 0)
                    sortInfoSize = firstRecordOffset - sortInfoOffset;
                else
                    sortInfoSize = statbuf.st_size - sortInfoOffset;

                if (read_sort_info(iprc, fp,
                                   sortInfoOffset, sortInfoSize) == -1)
                    return NULL;
            }
        }

        if (rv == -1)
            return NULL;
    }

    return &iprc->header;
}
