/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 * Copyright 1999 David Williams. All rights reserved.
 *
 * prc.h
 *
 * Defines the libprc library interface.
 *
 * created: djw@djw.org, January 2, 1999.
 *
 *------------------------------------
 *
 * Prc Format.
 * ==========
 * PrcHeader 78 bytes
 * ----------
 * SectionHeader0 10 bytes
 *  char name[4];
 *  short id;
 *  short offseth;
 *  short offsetl;
 * ----------
 * SectionHeader1 10 bytes
 * ----------
 * SectionHeaderN 10 bytes
 * ----------
 * SectionHeaderPadding 2 bytes
 * ----------
 * Data0
 * ----------
 * Data1
 * ----------
 * DataN
 * ----------
 *
 * Pdb Format.
 * ==========
 * PrcHeader 78 bytes
 * ----------
 * SectionHeader0 8 bytes
 *  short offseth;
 *  short offsetl;
 *  long  flags;
 * ----------
 * SectionHeader1 8 bytes
 * ----------
 * SectionHeaderN 8 bytes
 * ----------
 * SectionHeaderPadding 2 bytes
 * ----------
 * [AppInfo]
 * ----------
 * [SortInfo]
 * ----------
 * Data0
 * ----------
 * Data1
 * ----------
 * DataN
 * ----------
 */
#ifndef _PRC_H_
#define _PRC_H_

#define LIBPRC_VERSION       0x00000300L /* 00MMmmuu = MM.mm[.uu] */

/*
 * Version history:
 * 00000100 First versioned version.
 * 00000200 Delete, app, sort info support, par touch options, Win32.
 * 00000300 Bug fixes: delete multiple records, leaks in prcclose(),
 *          correct truncation in delete mode, record set bugs.
 *          New: support for new 3.X database attributes, db append.
 *          Win32 wildcard support added. Killed strptime()/strftime()
 */
#define LIBPRC_VERSION_MAJOR ((LIBPRC_VERSION>>16)&0xff)
#define LIBPRC_VERSION_MINOR ((LIBPRC_VERSION>>8)&0xff)
#define LIBPRC_VERSION_MICRO ((LIBPRC_VERSION)&0xff)

/* types, these should be ported to any new host */
typedef unsigned char  prc_byte_t;
typedef short          prc_int16_t;
typedef unsigned short prc_uint16_t;
typedef long           prc_int32_t;
typedef unsigned long  prc_uint32_t;
typedef unsigned long  prc_time_t;
typedef unsigned long  prc_type_t;
typedef prc_uint16_t   prc_attr_t;

#define PRC_MAX_NAMESIZE 32

/* File header */
typedef struct prc_t {
    prc_byte_t   name[PRC_MAX_NAMESIZE];
    prc_attr_t   flags;
    prc_uint16_t version;
    prc_time_t   ctime;
    prc_time_t   mtime;
    prc_time_t   btime;
    prc_uint32_t modnum;
    prc_uint32_t type;
    prc_uint32_t cid;
    prc_uint16_t nrecords;
    void*        appinfoData;
    prc_uint32_t appinfoSize;
    void*        sortinfoData;
    prc_uint32_t sortinfoSize;
} prc_t;

/* File attributes */
#define PRC_FLAGS_RESOURCE (0x1<<0)
#define PRC_FLAGS_READONLY (0x1<<1)
#define PRC_FLAGS_DIRTY    (0x1<<2)
#define PRC_FLAGS_BACKUP   (0x1<<3)
#define PRC_FLAGS_NEWER    (0x1<<4) /* PalmOS2 */
#define PRC_FLAGS_RESET    (0x1<<5) /* PalmOS2 */
#define PRC_FLAGS_COPYPREVENT (0x1<<6) /* PalmOS3 */
#define PRC_FLAGS_STREAM      (0x1<<7) /* PalmOS3 */
#define PRC_FLAGS_HIDDEN      (0x1<<8) /* PalmOS3.2 */
#define PRC_FLAGS_LAUNCHABLE  (0x1<<9) /* PalmOS3.2 */
#define PRC_FLAGS_OPEN     (0x1<<15)

#define PRC_IS_RESOURCE(p) (((p)->flags & PRC_FLAGS_RESOURCE) != 0)

/* Record Attributes */
#define	PRC_REC_FLAGS_DELETE 0x80
#define	PRC_REC_FLAGS_DIRTY  0x40
#define	PRC_REC_FLAGS_BUSY   0x20
#define	PRC_REC_FLAGS_SECRET 0x10
#define PRC_REC_FLAGS_GET_CATEGORY(x) ((x) & 0xf)
#define PRC_REC_FLAGS_MAKE(f,c) (((c) & 0xf)|((f) & 0xf0))

/* Record header */
typedef struct prc_record_t {
    prc_byte_t*  data;
    prc_uint32_t datalen; /* needs to be 32 bit even though device chokes */
    prc_uint32_t id;
    prc_byte_t   flags;
} prc_record_t;

/* Resource header */
typedef struct prc_resource_t {
    prc_byte_t*  data;
    prc_uint32_t datalen;
    prc_uint32_t type;
    prc_uint16_t id;
} prc_resource_t;

#define PRC_OPEN_READ   (0)
#define PRC_OPEN_WRITE  (0x1<<0)
#define PRC_OPEN_CREATE (0x1<<1)
#define PRC_OPEN_TOUCH  (0x1<<2) /* will not flush records */
#define PRC_OPEN_UPDATE (0x1<<3)

/* database manipulation */
prc_t*        prccreate(const char* fname,
                        const char* name,
                        prc_type_t  type,
                        prc_type_t  cid,
                        prc_attr_t  attrs);
/* convenience */
#define prccreateprc(f,n,t,c) prccreate((f),(n),(t),(c),PRC_FLAGS_RESOURCE);
#define prccreatepdb(f,n,t,c) prccreate((f),(n),(t),(c),0);

prc_t*        prcopen(const char* name, unsigned mode);
prc_t*        prcfpopen(FILE* fp, unsigned mode);
int           prcclose(prc_t*);
int           prcflush(prc_t*);
int           prcdestroy(prc_t* prc);
int           prcsetappinfo(prc_t*,
                            prc_byte_t*  data,
                            prc_uint16_t datalen);
int           prcsetsortinfo(prc_t*,
                             prc_byte_t*  data,
                             prc_uint16_t datalen);

/* record interface */
unsigned      prcgetnrecords(prc_t*);
prc_record_t* prcgetrecord(prc_t*, prc_uint16_t index);
prc_record_t* prcnewrecord();
int           prcdestroyrecord(prc_record_t* rec);
prc_record_t* prcremoverecord(prc_t* prc, prc_record_t* rec);
int           prcrecordset(prc_record_t*,
                           prc_byte_t*  data,
                           prc_uint16_t datalen);
prc_record_t* prcappendrecord(prc_t*, prc_record_t*);
prc_record_t* prcgetnextrecord(prc_t* prc, prc_record_t* last);
typedef int (*prc_record_mappee_t)(prc_record_t* record, void* arg);
int           prcmaprecords(prc_t* prc, prc_record_mappee_t, void* arg);

/* resource interface */
prc_resource_t* prcgetresource(prc_t*,
                               prc_uint32_t type,
                               prc_uint16_t id);
prc_resource_t* prcnewresource();
int             prcdestroyresource(prc_resource_t* resource);
prc_resource_t* prcremoveresource(prc_t* prc, prc_resource_t* resource);
int             prcresourceset(prc_resource_t*,
                               prc_uint32_t type,
                               prc_uint16_t id,
                               prc_byte_t*  data,
                               prc_uint16_t datalen);
prc_resource_t* prcappendresource(prc_t*, prc_resource_t*);
prc_resource_t* prcgetnextresource(prc_t* prc, prc_resource_t* last);
typedef int (*prc_resource_mappee_t)(prc_resource_t* resource, void* arg);
int           prcmapresources(prc_t* prc, prc_resource_mappee_t, void* arg);

/* type/cid util */
char*      prctypetostr(char* buf, prc_type_t type);
prc_type_t prcstrtotype(const char* buf);

/* time util */
prc_time_t  prctime(prc_time_t* buf); /* now */
char*       prctimetostr(char* buf, prc_time_t prc_time);
prc_time_t  prcstrtotime(const char*);

/* file attribute util */
char*       prcstrfattr(char* buf, prc_attr_t flags);
prc_attr_t  prcstrpattr(const char* buf);

/* misc useful utils */
void        prcsetname(prc_t* prc, const char* name);

#endif /*_PRC_H_*/
